// drawtool.cpp - implementation for drawing tools
//
// This is a part of the Microsoft Foundation Classes C++ library.
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
// This source code is only intended as a supplement to the
// Microsoft Foundation Classes Reference and related
// electronic documentation provided with the library.
// See these sources for detailed information regarding the
// Microsoft Foundation Classes product.


#include "stdafx.h"
#include "SavApp.h"
#include "SavDoc.h"
#include "SavView.h"
#include "drawobj.h"
#include "drawtool.h"

#include <math.h>

/////////////////////////////////////////////////////////////////////////////
// CDrawTool implementation

CPtrList CDrawTool::c_tools;

static CSelectTool selectTool;
static CRectTool lineTool(line);
static CRectTool ellipseTool(ellipse);

CPoint CDrawTool::c_down;
UINT CDrawTool::c_nDownFlags;
CPoint CDrawTool::c_last;
DrawShape CDrawTool::c_drawShape = selection;

CDrawTool::CDrawTool(DrawShape drawShape)
{
	m_drawShape = drawShape;
	c_tools.AddTail(this);
}

CDrawTool* CDrawTool::FindTool(DrawShape drawShape)
{
	POSITION pos = c_tools.GetHeadPosition();
	while (pos != NULL)
	{
		CDrawTool* pTool = (CDrawTool*)c_tools.GetNext(pos);
		if (pTool->m_drawShape == drawShape)
			return pTool;
	}

	return NULL;
}

void CDrawTool::OnLButtonDown(CSavView* pView, UINT nFlags, const CPoint& point)
{
	// deactivate any in-place active item on this view!
	COleClientItem* pActiveItem = pView->GetDocument()->GetInPlaceActiveItem(pView);
	if (pActiveItem != NULL)
	{
		pActiveItem->Close();
		ASSERT(pView->GetDocument()->GetInPlaceActiveItem(pView) == NULL);
	}

	pView->SetCapture();
	c_nDownFlags = nFlags;
	c_down = point;
	c_last = point;
}

void CDrawTool::OnLButtonDblClk(CSavView* /*pView*/, UINT /*nFlags*/, const CPoint& /*point*/)
{
}

void CDrawTool::OnLButtonUp(CSavView* /*pView*/, UINT /*nFlags*/, const CPoint& point)
{
	ReleaseCapture();

	if (point == c_down)
		c_drawShape = selection;
}

void CDrawTool::OnMouseMove(CSavView* /*pView*/, UINT /*nFlags*/, const CPoint& point)
{
	c_last = point;
	SetCursor(AfxGetApp()->LoadStandardCursor(IDC_ARROW));
}

void CDrawTool::OnEditProperties(CSavView* /*pView*/)
{
}

void CDrawTool::OnCancel()
{
	c_drawShape = selection;
}

////////////////////////////////////////////////////////////////////////////
// CResizeTool

enum SelectMode
{
	none,
	netSelect,
	move,
	size
};

SelectMode selectMode = none;
int nDragHandle;

CPoint lastPoint;

CSelectTool::CSelectTool()
	: CDrawTool(selection)
{
}

void CSelectTool::OnLButtonDown(CSavView* pView, UINT nFlags, const CPoint& point)
{
	CPoint local = point;
	pView->ClientToDoc(local);

	CDrawObj* pObj;
	selectMode = none;

	// Check for resizing (only allowed on single selections)
	if (pView->m_selection.GetCount() == 1)
	{
		pObj = pView->m_selection.GetHead();
		nDragHandle = pObj->HitTest(local, pView, TRUE);
		if (nDragHandle != 0)
			selectMode = size;
	}

	// See if the click was on an object, select and start move if so
	if (selectMode == none)
	{
		pObj = pView->GetDocument()->ObjectAt(local);

		if (pObj != NULL)
		{
			selectMode = move;

			if (!pView->IsSelected(pObj))
				pView->Select(pObj, (nFlags & MK_SHIFT) != 0);

			// Ctrl+Click clones the selection...
			if ((nFlags & MK_CONTROL) != 0)
				pView->CloneSelection();
		}
	}

	// Click on background, start a net-selection
	if (selectMode == none)
	{
		if ((nFlags & MK_SHIFT) == 0)
			pView->Select(NULL);

		selectMode = netSelect;

		CClientDC dc(pView);
		CRect rect(point.x, point.y, point.x, point.y);
		rect.NormalizeRect();
		dc.DrawFocusRect(rect);
	}

	lastPoint = local;
	CDrawTool::OnLButtonDown(pView, nFlags, point);
}

void CSelectTool::OnLButtonDblClk(CSavView* pView, UINT nFlags, const CPoint& point)
{
	if ((nFlags & MK_SHIFT) != 0)
	{
		// Shift+DblClk deselects object...
		CPoint local = point;
		pView->ClientToDoc(local);
		CDrawObj* pObj = pView->GetDocument()->ObjectAt(local);
		if (pObj != NULL)
			pView->Deselect(pObj);
	}
	else
	{
		// "Normal" DblClk opens properties, or OLE server...
		if (pView->m_selection.GetCount() == 1)
			pView->m_selection.GetHead()->OnOpen(pView);
	}

	CDrawTool::OnLButtonDblClk(pView, nFlags, point);
}

void CSelectTool::OnEditProperties(CSavView* pView)
{
	if (pView->m_selection.GetCount() == 1)
		pView->m_selection.GetHead()->OnEditProperties();
}

void CSelectTool::OnLButtonUp(CSavView* pView, UINT nFlags, const CPoint& point)
{
	if (pView->GetCapture() == pView)
	{
		if (selectMode == netSelect)
		{
			CClientDC dc(pView);
			CRect rect(c_down.x, c_down.y, c_last.x, c_last.y);
			rect.NormalizeRect();
			dc.DrawFocusRect(rect);

			pView->SelectWithinRect(rect, TRUE);
		}
		else if (selectMode != none)
		{
			//
			// ְ ߰
			//
			/*
			CDrawRect* pObj = (CDrawRect*)pView->m_selection.GetHead();
			if (pObj && pObj->getShape() == CDrawRect::ellipse) {
				//pObj = new CDrawRect(CRect(local, CSize(0, 0)));
				pObj->getCenterPos();
			}
			*/

			pView->GetDocument()->UpdateAllViews(pView);
		}
	}

	CDrawTool::OnLButtonUp(pView, nFlags, point);
}

void CSelectTool::OnMouseMove(CSavView* pView, UINT nFlags, const CPoint& point)
{
	if (pView->GetCapture() != pView)
	{
		if (c_drawShape == selection && pView->m_selection.GetCount() == 1)
		{
			CDrawObj* pObj = pView->m_selection.GetHead();
			CPoint local = point;
			pView->ClientToDoc(local);
			int nHandle = pObj->HitTest(local, pView, TRUE);
			if (nHandle != 0)
			{
				SetCursor(pObj->GetHandleCursor(nHandle));
				return; // bypass CDrawTool
			}
		}
		if (c_drawShape == selection)
			CDrawTool::OnMouseMove(pView, nFlags, point);
		return;
	}

	if (selectMode == netSelect)
	{
		CClientDC dc(pView);
		CRect rect(c_down.x, c_down.y, c_last.x, c_last.y);
		rect.NormalizeRect();
		dc.DrawFocusRect(rect);
		rect.SetRect(c_down.x, c_down.y, point.x, point.y);
		rect.NormalizeRect();
		dc.DrawFocusRect(rect);

		CDrawTool::OnMouseMove(pView, nFlags, point);
		return;
	}

	CPoint local = point;
	pView->ClientToDoc(local);
	CPoint delta = (CPoint)(local - lastPoint);

	POSITION pos = pView->m_selection.GetHeadPosition();
	while (pos != NULL)
	{
		CDrawObj* pObj = pView->m_selection.GetNext(pos);
		CRect position = pObj->m_position;

		if (selectMode == move)
		{
			position += delta;
			pObj->MoveTo(position, pView);
		}
		else if (nDragHandle != 0)
		{
			pObj->MoveHandleTo(nDragHandle, local, pView);
		}
	}

	lastPoint = local;

	if (selectMode == size && c_drawShape == selection)
	{
		c_last = point;
		SetCursor(pView->m_selection.GetHead()->GetHandleCursor(nDragHandle));
		return; // bypass CDrawTool
	}

	c_last = point;

	if (c_drawShape == selection)
		CDrawTool::OnMouseMove(pView, nFlags, point);
}

////////////////////////////////////////////////////////////////////////////
// CRectTool (does rectangles, round-rectangles, and ellipses)

CRectTool::CRectTool(DrawShape drawShape)
	: CDrawTool(drawShape)
{
}

void CRectTool::OnLButtonDown(CSavView* pView, UINT nFlags, const CPoint& point)
{
	CDrawTool::OnLButtonDown(pView, nFlags, point);
	CPoint local;
	
	CDrawRect* pObj;
	switch (m_drawShape)
	{
	default:
		ASSERT(FALSE); // unsuported shape!

	case ellipse:
		local = CPoint(point.x-50, point.y+50);
		pView->ClientToDoc(local);
		pObj = new CDrawRect(CRect(local, CSize(100,100)));
		pObj->m_nShape = CDrawRect::ellipse;
		break;

	case line:
		local = point;
		pView->ClientToDoc(local);
		pObj = new CDrawRect(CRect(local, CSize(0, 0)));
		pObj->m_nShape = CDrawRect::line;
		break;
	}
	pView->GetDocument()->Add(pObj);
	pView->Select(pObj);

	selectMode = size;
	nDragHandle = 1;
	lastPoint = local;
}

void CRectTool::OnLButtonDblClk(CSavView* pView, UINT nFlags, const CPoint& point)
{
	CDrawTool::OnLButtonDblClk(pView, nFlags, point);
}

void CRectTool::OnLButtonUp(CSavView* pView, UINT nFlags, const CPoint& point)
{
	if (m_drawShape == line) {
		//
		//      ٸ ellipse ȿ ִ ȮѴ.
		// 
		CPoint down = c_down;
		CPoint up = point;
		CDrawObj* p1 = pView->findEllipse(down);
		CDrawObj* p2 = pView->findEllipse(up);
		
		pView->removeDrawObj(NULL);
		if (p1 && p2) {
			//drawArrowLine(pView, p1->getCenterPos(), p2->getCenterPos());
			drawArrowLine(pView, p1, p2);
		}
	}

	selectTool.OnLButtonUp(pView, nFlags, point);
}

void CRectTool::OnMouseMove(CSavView* pView, UINT nFlags, const CPoint& point)
{
	if (m_drawShape == line) {
		SetCursor(AfxGetApp()->LoadStandardCursor(IDC_CROSS));
		selectTool.OnMouseMove(pView, nFlags, point);
	}
}

//
//   ʿ䰡 ִ Լ
//
void CRectTool::drawEllipse(CSavView* pView, CPoint& r)
{
	pView->ClientToDoc(r);
	CDrawRect* pObj = new CDrawRect(CRect(r, CSize(100,100)));
	pObj->m_nShape = CDrawRect::ellipse;
}

void CRectTool::drawArrowLine(CSavView* pView, CDrawObj* line_from, CDrawObj* line_to)
{
	CPoint r1, r2;
	r1 = line_from->getCenterPos();
	r2 = line_to->getCenterPos();

	double slopy , cosy , siny;
	slopy = atan2( (double)( r2.y - r1.y ),
		(double)( r2.x - r1.x ) );
	cosy = cos( slopy );siny = sin( slopy );
	
	CPoint rp1,rp2;
	rp2.x = (long)(r2.x-50*cosy);rp2.y = (long)(r2.y-50*siny);
	rp1.x = (long)(r1.x+50*cosy);rp1.y = (long)(r1.y+50*siny);

	CDrawRect* newLine = new CDrawRect(CRect(rp2, rp1), 
		(CDrawRect*)line_from, (CDrawRect*)line_to);
	newLine->m_nShape = CDrawRect::line;
	pView->GetDocument()->Add(newLine);
	pView->Select(newLine);
}

void CRectTool::drawArrowLine(CSavView* pView, CPoint& r1, CPoint& r2)
{
	double slopy , cosy , siny;
	slopy = atan2( (double)( r2.y - r1.y ),
		(double)( r2.x - r1.x ) );
	cosy = cos( slopy );siny = sin( slopy );
	
	CPoint rp1,rp2;
	rp2.x = (long)(r2.x-50*cosy);rp2.y = (long)(r2.y-50*siny);
	rp1.x = (long)(r1.x+50*cosy);rp1.y = (long)(r1.y+50*siny);

	CDrawRect* newLine = new CDrawRect(CRect(rp2, rp1));
	newLine->m_nShape = CDrawRect::line;
	pView->GetDocument()->Add(newLine);
	pView->Select(newLine);
}
/////////////////////////////////////////////////////////////////////////////
