/*
 * CDIC by YoungJoo-Kim <bando@bando.org>
 * 
 * 
 * <http://cdic.kldp.net>
 * <http://kldp.net/projects/cdic>
 *    
 *  
 * Compile
 * [bando@bando cdic-1.0]$ gcc -o cdic cdic.c 
 * 
 * 
 * Usage
 * [bando@bando cdic-1.0]$ ./cdic [word] or [text]
 * 
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <netdb.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <pwd.h>

#ifdef HAVE_ICONV
#include <iconv.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>

#define PROG_NAME "cdic"
#define PROG_VERSION "1.0"
#define CONF_F ".cdic"

#define SUCCESS 0
#define FAILURE 1

#define BLOSOCK 1
#define NONSOCK 0
#define LINGERON 1
#define LINGEROFF 0
#define SOTIMEO 44
#define WFI 10

#define BUFSIZE 1024

#ifdef HAVE_ICONV
#define CDSET "EUC-KR"
#endif

#define IS_INVALID_SOCKET(sock) (sock < 0)
#define IS_SOCKET(sock) (sock > 0)

#define _COUNT(arr, type) (sizeof(arr)/sizeof(type))

typedef enum _dic_type dic_type;
typedef enum _trans_type trans_type;
#ifdef HAVE_ICONV
typedef enum _iconv_err_type iconv_err_type;
#endif

enum _dic_type { NAVER, YAHOO, DAUM, EMPAS };
enum _trans_type { GOOGLE_TRANS, YAHOO_TRANS };
enum _ANSI { GRAY = 30, RED, GREEN, YELLOW, BLUE, MAGENTA, CYAN, WHITE, NOT = 00, BOLD = 1, UNBOLD = 0 };
#ifdef HAVE_ICONV
enum _iconv_err_type {
	ICONV_ERR_SUCCESS = SUCCESS,
	ICONV_ERR_CONVERTER,
	ICONV_ERR_WRONG_CHARSET,
	ICONV_ERR_TOO_BIG,
	ICONV_ERR_ILLEGAL_SEQ,
	ICONV_ERR_ILLEGAL_CHAR,
	ICONV_ERR_UNKNOWN
};
#endif

typedef struct _config {
	dic_type dic;
	trans_type trans;
#ifdef HAVE_ICONV
	char charset[64];
#endif
} config;

typedef void (*dic_func)(char *str, config *cf);
typedef void (*trans_func)(char *str, config *cf);

/*
enum _dic_type { NAVER, YAHOO, DAUM, EMPAS };
enum _trans_type { GOOGLE_TRANS, YAHOO_TRANS };
enum _ANSI { GRAY = 30, RED, GREEN, YELLOW, BLUE, MAGENTA, CYAN, WHITE, NOT = 00, BOLD = 1, UNBOLD = 0 };

#ifdef HAVE_ICONV
enum _iconv_err_type {
	ICONV_ERR_SUCCESS = SUCCESS,
	ICONV_ERR_CONVERTER,
	ICONV_ERR_WRONG_CHARSET,
	ICONV_ERR_TOO_BIG,
	ICONV_ERR_ILLEGAL_SEQ,
	ICONV_ERR_ILLEGAL_CHAR,
	ICONV_ERR_UNKNOWN
};
#endif
*/

typedef struct _dic_cmd {
	char *type_name;
	dic_type type;
	dic_func func;
} dic_cmd;

typedef struct _trans_cmd {
	char *type_name;
	trans_type type;
	trans_func func;
} trans_cmd;

typedef struct _http_request {
	char *host;
	char *uri;
	char *rhdr;
	unsigned int port;
} http_request;


/*
typedef struct _config {
	dic_type dic;
	trans_type trans;
#ifdef HAVE_ICONV
	char charset[64];
#endif
} config;
*/

int socket_get_option (int sock, int level, int optname, void *optval);
int socket_set_option (int sock, int level, int optname, const void *optval);
int socket_set_blocking (int sock, int block);
void socket_close (int sock, int l_onoff);
int tcp_connect (char *server, unsigned int port, unsigned int timeout);
int read_line (char *str, size_t size, int fd);
char *read_all (int fd);
int write_all (int fd, const void *buf, size_t count);
int wait_for_input (int fd, unsigned int seconds);
int parse_url (const char *url, http_request *hr);
char *split (char *str, char split, int ret);
char *space_encode(char *str, char *ret, int ret_len);
int http_send_get (int sock, http_request *hr);
char *http_recv_data (int sock);
int http_recv_code (int sock, char *replycode);
int str_cut (char *str, char *start, char *end);
int str_toupper (char *str);
int str_tolower (char *str);
int str_trim (char *str);
int str_trims (char *str);
int str_replace (char *str, char *now, char *new);
int strip_tag (char *str);

#ifdef HAVE_ICONV
void iconv_error (const char *in_charset, const char *out_charset, iconv_err_type iconv_errno);
iconv_err_type iconv_convert (char *in_charset, char *out_charset, char *in_str, char **out_str);
#endif

void err_exit (char *fmt, ...);
void http_request_free (http_request *hr);
void http_request_init (http_request *hr);
void usage (const char *prog, config *cf);
char *ansi_color (char *str, unsigned int color, unsigned int bold);
void dic_naver (char *str, config *cf);
void dic_yahoo (char *str, config *cf);
void dic_daum (char *str, config *cf);
void dic_empas (char *str, config *cf);
void trans_google (char *str, config *cf);
void trans_yahoo (char *str, config *cf);
void config_init (config *cf);
int config_load (char *path, config *cf);
int config_reload (char *var, config *cf);
int config_create (char *path, config *cf);

dic_cmd dic[] =
{
        { "NAVER", NAVER, dic_naver },
        { "YAHOO", YAHOO, dic_yahoo },
        { "DAUM", DAUM, dic_daum },
        { "EMPAS", EMPAS, dic_empas }
};

trans_cmd trans[] =
{
        { "GOOGLE_TRANS", GOOGLE_TRANS, trans_google },
        { "YAHOO_TRANS", YAHOO_TRANS, trans_yahoo }
};

int main (int argc, char **argv)
{
	struct passwd *pw_entry;
	config cf;
	char buf[BUFSIZE], *result = NULL;
	int i = 0;
	
	memset(buf, 0x0, sizeof(buf));
	pw_entry = getpwuid(getuid());
	snprintf(buf, sizeof(buf) - 1, "%s/%s", pw_entry->pw_dir, CONF_F);
	config_load(buf, &cf);
	if (argc < 2)
		usage(PROG_NAME, &cf);

	else if (argc == 2) {
		if (strchr(argv[1], ' ')) {
			trans[cf.trans].func(argv[1], &cf);
		} else if (strchr(argv[1], '=')) {
			if (config_reload (argv[1], &cf) == SUCCESS) {
				//printf(" ˻   => %s : ", strchr(argv[1], '=') + 1);
				result = (config_create (buf, &cf) == SUCCESS) ? ansi_color("", GREEN, UNBOLD) : ansi_color("", RED, UNBOLD);
				printf("  [%s]\n", result);
				if (result) free(result);
			} else { dic[cf.dic].func(argv[1], &cf); }
		} else {
			dic[cf.dic].func(argv[1], &cf);
		}
	} else {
		memset(buf, 0x0, sizeof(buf));
		for (i = 0; ++i < argc;) {
			strncat(buf, argv[i], strlen(argv[i]));
			if ((argc - 1) > i) strncat(buf, " ", sizeof(char));
		}
		trans[cf.trans].func(buf, &cf);
	}

	return 0;
}

int socket_get_option (int sock, int level, int optname, void *optval)
{
	socklen_t optlen;

	switch(optname) {
		case SO_LINGER:
			optlen = sizeof(struct linger);
			if (getsockopt(sock, level, optname, optval, &optlen) != 0) {
				perror("unable to retrieve socket option");
				return FAILURE;
			}
			break;

		case SO_RCVTIMEO:
		case SO_SNDTIMEO:
			optlen = sizeof(struct timeval);
			if (getsockopt(sock, level, optname, optval, &optlen) != 0) {
				perror("unable to retrieve socket option");
				return FAILURE;
			}
			break;

		default:
			optlen = sizeof(int);
			if (getsockopt(sock, level, optname, optval, &optlen) != 0) {
				perror("unable to retrieve socket option");
				return FAILURE;;
			}
			break;
	}
	return SUCCESS;
}

int socket_set_option (int sock, int level, int optname, const void *optval)
{
	socklen_t optlen;

	switch(optname) {
		case SO_LINGER:
			optlen = sizeof(struct linger);
			if (setsockopt(sock, level, optname, optval, optlen) != 0) {
				perror("unable to retrieve socket option");
				return FAILURE;
			}
			break;

		case SO_RCVTIMEO:
		case SO_SNDTIMEO:
			optlen = sizeof(struct timeval);
			if (setsockopt(sock, level, optname, optval, optlen) != 0) {
				perror("unable to retrieve socket option");
				return FAILURE;
			}
			break;

		default:
			optlen = sizeof(int);
			if (setsockopt(sock, level, optname, optval, optlen) != 0) {
				perror("unable to retrieve socket option");
				return FAILURE;;
			}
			break;
	}
	return SUCCESS;
}

int socket_set_blocking (int sock, int block)
{
	int flags;
	int myflag = 0;

	flags = fcntl(sock, F_GETFL, 0);
	myflag = O_NONBLOCK;

	if (!block) {
		flags |= myflag;
	} else {
		flags &= ~myflag;
	}
	
	return fcntl(sock, F_SETFL, flags);
}

void socket_close (int sock, int l_onoff)
{
	struct linger linger_v;		

	if (l_onoff > 0) {
		linger_v.l_onoff = 0x01;
		linger_v.l_linger = 0x00;
		socket_set_option(sock, SOL_SOCKET, SO_LINGER, &linger_v);
	}
	close(sock);
}

int tcp_connect (char *server, unsigned int port, unsigned int timeout)
{
	struct sockaddr_in localAddr, servAddr;
	struct hostent *h;
	struct timeval timeval_v;
	int rc;
	int sock;

	h = gethostbyname(server);
	if (h == NULL)
		return -1;

	servAddr.sin_family = h->h_addrtype;
	memcpy((char *)&servAddr.sin_addr.s_addr, h->h_addr_list[0], h->h_length);
	servAddr.sin_port = htons(port);

	sock = socket(AF_INET, SOCK_STREAM, 0);
	if (sock < 0)
		return -1;

	if (timeout > 0) {
		timeval_v.tv_sec = timeout;
		timeval_v.tv_usec = 0;
		socket_set_option(sock, SOL_SOCKET, SO_SNDTIMEO, &timeval_v);
		socket_set_option(sock, SOL_SOCKET, SO_RCVTIMEO, &timeval_v);
	}

	localAddr.sin_family = AF_INET;
	localAddr.sin_addr.s_addr = htonl(INADDR_ANY);
	localAddr.sin_port = htons(0);

	rc = bind(sock, (struct sockaddr *)&localAddr, sizeof(localAddr));
	if (rc < 0)
		return -1;

	rc = connect(sock, (struct sockaddr *)&servAddr, sizeof(servAddr));
	if (rc < 0)
		return -1;

	return sock;
}

int read_line (char *str, size_t size, int fd)
{
	char c, *ptr = str;
	int r;

	while ((r = read(fd, &c, 1)) != 0) {
		if (r < 0)
			return -1;

		if ((ptr - str) >= size)
			break; 

		*(ptr++) = c;

		if (c == '\n')
			break;
	}

	*ptr = '\0';
	return (ptr - str);
} 

char *read_all (int fd)
{
	char c, *ptr = NULL;
	int r, i;
	i = 0;
	
	ptr =  (char *)calloc((i + 1), sizeof(char));

	while (wait_for_input(fd, WFI) > 0) {
		if ((r = read(fd, &c, 1)) > 0) {
			ptr = (char *)realloc(ptr, sizeof(char) * (i + 1));	
			*(ptr + i) = c;
			i++;
		} else if (r == 0) {
			break;
		}
		else {
			return (char *)NULL;
		}
	}
	ptr = (char *)realloc(ptr, sizeof(char) * (i + 1));	
	*(ptr + i) = '\0';

	return ptr;
}

int write_all (int fd, const void *buf, size_t count)
{
	int res, remaining, offset;

	for (remaining = count, offset = 0;
			remaining > 0;
			remaining -= res, offset += res)
	{
		res = write(fd, buf + offset, remaining);

		if (res == -1)
			return -1;
	}

	return count;
}

int wait_for_input (int fd, unsigned int seconds)
{
	fd_set set;
	struct timeval timeout;

	FD_ZERO(&set);
	FD_SET(fd, &set);

	timeout.tv_sec = seconds;
	timeout.tv_usec = 0;

	return select(fd + 1, &set, (fd_set *)NULL, (fd_set *)NULL, &timeout);
}

int parse_url (const char *url, http_request *hr)
{
	char *colon = NULL;
	char delim[]="://";
	char *cpstart = NULL;
	char *cpend = NULL;

	colon = strstr(url, delim);

	if (colon == NULL)
		return FAILURE;

	cpstart = colon + strlen(delim);
	cpend = strchr(colon + strlen(delim),'/');

	if (cpend == NULL)
		return FAILURE;

	hr->host = (char *)calloc(cpend - cpstart + 1 , sizeof(char));

	strncpy(hr->host, cpstart, cpend - cpstart);

	hr->host[cpend - cpstart] = '\0';

	hr->uri = (char *)calloc(url + strlen(url) - cpend + 1 , sizeof(char));

	strncpy(hr->uri, cpend, strlen(cpend));

	hr->uri[strlen(cpend)] = '\0';

	if  ((colon = strchr( hr->host, ':' )) == NULL) {
		hr->port = 80;
	} else {
		hr->port = atoi(colon + 1);
		hr->host[colon - hr->host] = '\0';
	}

	return SUCCESS;
}

char *split (char *str, char split, int ret)
{
	char *ptr = NULL;
	int a, b, c = 0;

	if (ret != 0) {
		for (a = 0; a <= strlen(str); a++) {
			if (str[a] == split && ++c == ret) {
				a++;
				break;
			}
			if (a == strlen(str))
				return NULL;
		}
	} else {
		a = 0;
	}

	for (b = a; b <= strlen(str); b++)
		if (str[b] == split)
			break;

	ptr = (char *)calloc(sizeof(char) * (b - a + 1), sizeof(char));
	strncpy(ptr, &str[a], b - a);
	ptr[b - a] = 0x0;

	return (char *)ptr;
}

char *space_encode(char *str, char *ret, int ret_len)
{
	int i, a;

	for (i = 0, a = 0; i < strlen(str) && a < ret_len; i++, a++) {
		if (isspace(str[i])) {
			snprintf(&ret[a], ret_len - i, "%%20");
			a += 2;
		} else {
			ret[a] = str[i];
		}
	}
	ret[a] = 0x0;
	return ret;
}

int http_send_get (int sock, http_request *hr)
{
	char buf[BUFSIZE];

	space_encode(hr->uri, buf, sizeof(buf) - 1);

	hr->rhdr = (char *) calloc(BUFSIZE<<1, sizeof(char));

	snprintf(hr->rhdr, (BUFSIZE<<1) - 1,
			"GET %s HTTP/1.0\r\n"
			"Host: %s\r\n"
			"User-Agent: Cdic/%s\r\n"
			"Accept: */*\r\n"
			"Connection: close\r\n\r\n"
			, buf, hr->host, PROG_VERSION
	);

	if (write_all(sock, hr->rhdr, strlen(hr->rhdr)) < 0)
	{
		socket_close(sock, LINGEROFF);
		return FAILURE;
	}

	return SUCCESS;
}

char *http_recv_data (int sock)
{
	int res;
	char ch;
	char *recved = NULL;

	while (wait_for_input(sock, WFI) > 0)
	{
		res = read(sock, &ch, 1);
		if (res == -1 || res == 0)
			return (char *)NULL;

		if (ch != '\r')
			continue;	

		res = read(sock, &ch, 1);
		if (res == -1 || res == 0)
			return (char *)NULL;

		if (ch == '\n') {
			res = read(sock, &ch, 1);
			if (res == -1 || res == 0)
				return (char *)NULL;

			if (ch != '\r')
				continue;

			res = read(sock, &ch, 1);
			if (res == -1 || res == 0)
				return (char *)NULL;

			if (ch == '\n') {
				recved = read_all(sock);
				break;
			}

		}
	}
	return recved;
}

int http_recv_code (int sock, char *replycode)
{
	int res, i = 0;
	char ch;
	char buf[BUFSIZE];

	while (wait_for_input(sock, WFI) > 0) {

		res = read(sock, &ch, 1);
		if (res == -1 || res == 0)
			return -1;

		if (ch != '\r') {
			buf[i++] = ch;
			continue;
		}

		res = read (sock, &ch, 1);
		if (res == -1 || res == 0)
			return -1;

		if (ch == '\n') {
			buf[i] = '\0';
			break;
		}
	}

	return (strstr(buf, replycode) == NULL) ? FAILURE : SUCCESS;
}

int str_cut (char *str, char *start, char *end)
{
	char *sp = NULL, *ep = NULL;
	int sp_len = strlen(start);

	sp = strstr(str, start);
	if (sp && strlen(sp) > strlen(start))
		ep = strstr(sp + strlen(start) -1, end);
	else return -1;

	if (sp && ep && ((ep - sp) > 0)) {
		memmove(str, sp + sp_len, ep - sp - sp_len);
		str[ep - sp - sp_len] = '\0';
	} else {
		return -1;
	}
	return (ep - sp - sp_len);
}

int str_toupper (char *str)
{
	int len, i = 0;
	len = strlen(str);

	while (i < len) {
		*(str + i) = toupper(*(str + i));
		i++;
	}

	return (i -1);
}

int str_tolower (char *str)
{
	int len, i = 0;
	len = strlen(str);

	while (i < len) {
		*(str + i) = tolower(*(str + i));
		i++;
	}

	return (i -1);
}

int str_trim (char *str)
{
	int sp = 0, ep = 0;
	int len, i = 0;
	len = strlen(str);

	for (i=0; i<len; i++) {
		if (!isspace(*(str + i))) {
			sp = i;
			break;
		}
	}
	for (i=0; i<len; i++) {
		if (!isspace(*(str + len - i -1))) {
			ep = i;
			break;
		}
	}
	memmove(str, str + sp, len - sp - ep);
	str[len - sp - ep] = '\0';
	
	return (sp + ep);
}

int str_trims (char *str)
{
	int len, i = 0, ln = 0;
	char *ptr = str;
	char *optr = str;
	char *lnbuf = NULL;
	char *nstr = NULL;

	len = strlen(str);

	for (i=0; i<len; i++) {
		if ('\n' == *(str + i)) {
			ln++;
		}
	}

	if (ln > 0) {
		lnbuf = (char *)calloc(len + 1, sizeof(char));
		nstr = (char *)calloc(len + 1, sizeof(char));

		while ((ptr = strchr(ptr, '\n')))
		{
			strncpy(lnbuf, optr, ptr - optr);
			*(lnbuf + (ptr - optr)) = '\0';
			str_trim(lnbuf);
			strncat(nstr, lnbuf, strlen(lnbuf));
			strcat(nstr, "\n");
			ptr++;
			optr = ptr;
		}
		if ( strlen(optr) > 0 ) {
			strncpy(lnbuf, optr, strlen(optr));
			*(lnbuf + strlen(optr)) = '\0';
			str_trim(lnbuf);
			strncat(nstr, lnbuf, strlen(lnbuf));

		}
		memcpy(str, nstr, strlen(nstr) + 1);
		free(lnbuf);
		free(nstr);
	} else {
		str_trim(str);
	}
	
	return ln;
}

int str_replace (char *str, char *now, char *new)
{
        int  str_len, now_len, new_len;
        int r = 0;
	char *ptr = str;
        str_len = strlen(str);
        now_len = strlen(now);
        new_len = strlen(new);

	if (strlen(now) > 0) {
		while ((ptr = strstr(ptr, now))) {
			ptr += now_len;
			r++;
		}

		if (now_len < new_len) {
			str = (char *)realloc(str, sizeof(char) * str_len + ((new_len - now_len) * r) + 1);
		}

		while ((str = strstr(str, now))) {
			memmove(str + new_len, str + now_len, strlen(str + now_len) + 1);
			memcpy(str, new, new_len);
			str += new_len;
		}
	}

	return r;
}

int strip_tag (char *str)
{
	char c, *ptr = NULL, *optr = NULL;
	int len, i = 0, ti = 0, state = 0, esc = 8;

	len = strlen(str);
	ptr =  (char *)calloc(len + 1, sizeof(char));
	optr = ptr;

	while (i < len) {

		c = *(str + i);

		switch (c) {
			case '\0':
				break;
			case '<':
				if (state == 0
						&& *(str + i + 1) == '!'
						&& *(str + i + 2) == '-'
						&& *(str + i + 3) == '-') {
					state = 1;
				} else if (state == 0
						&& tolower(*(str + i + 1)) == 's'
						&& tolower(*(str + i + 2)) == 'c'
						&& tolower(*(str + i + 3)) == 'r'
						&& tolower(*(str + i + 4)) == 'i'
						&& tolower(*(str + i + 5)) == 'p'
						&& tolower(*(str + i + 6)) == 't') {
					state = 2;
				} else if (state == 0
						&& tolower(*(str + i + 1)) == 's'
						&& tolower(*(str + i + 2)) == 't'
						&& tolower(*(str + i + 3)) == 'y'
						&& tolower(*(str + i + 4)) == 'l'
						&& tolower(*(str + i + 5)) == 'e') {
					state = 3;
				} else if (state == 0) {
					state = 4;
				}
				break;

			case '>':
				if (state == 1
						&& *(str + i - 1) == '-'
						&& *(str + i - 2) == '-') {
					state = 0;
				} else if (state == 2
						&& tolower(*(str + i - 7)) == '/'
						&& tolower(*(str + i - 6)) == 's'
						&& tolower(*(str + i - 5)) == 'c'
						&& tolower(*(str + i - 4)) == 'r'
						&& tolower(*(str + i - 3)) == 'i'
						&& tolower(*(str + i - 2)) == 'p'
						&& tolower(*(str + i - 1)) == 't') {
					state = 0;
				} else if (state == 3 
						&& tolower(*(str + i - 6)) == '/'
						&& tolower(*(str + i - 5)) == 's'
						&& tolower(*(str + i - 4)) == 't'
						&& tolower(*(str + i - 3)) == 'y'
						&& tolower(*(str + i - 2)) == 'l'
						&& tolower(*(str + i - 1)) == 'e') {
					state = 0;
				} else if (state == 4) {
					state = 0;
				}
				break;

			case '&':
				if (state == 0) { 
					for(ti = 1; ti < esc; ti++) {
						if (*(str + i + ti) == ';') {
							state = 0;
							i += (ti);
							break;
						}
					}
				}
				break;
			
			default:
				if (state == 0) {
					*(ptr++) = c;
				} 

				break;
		}
		i++;
	}
	strncpy(str, optr, strlen(optr) + 1);
	free(optr);
	
	return (len > strlen(str)) ? 0 : 1;
}

#ifdef HAVE_ICONV
void iconv_error (const char *in_charset, const char *out_charset, iconv_err_type iconv_errno)
{
	switch (iconv_errno) {
		case ICONV_ERR_SUCCESS:
			break;

		case ICONV_ERR_CONVERTER:
			err_exit("Cannot open converter\n");
			break; 

		case ICONV_ERR_WRONG_CHARSET:
			err_exit(
				"Wrong charset, conversion from `%s' to `%s' is not allowed\n"
				"The iconv -l or iconv --list command lists the names of the supported encodings, in a system dependent format\n"
				, in_charset, out_charset);
			break;  

		case ICONV_ERR_ILLEGAL_CHAR:
			err_exit("Detected an incomplete multibyte character in input string\n");
			break;  

		case ICONV_ERR_ILLEGAL_SEQ:
			err_exit("Detected an illegal character in input string\n");
			break;

		case ICONV_ERR_TOO_BIG:
			/* should not happen */
			err_exit("Buffer length exceeded\n");
			break;

		default:
			/* other error */
			err_exit("Unknown error (%d)\n", iconv_errno);
			break;
	}
}

iconv_err_type iconv_convert (char *in_charset, char *out_charset, char *in_str, char **out_str)
{
	iconv_t cd;
	size_t in_left, out_size, out_left;
	char *in_p, *out_p, *out_buf, *tmp_buf;
	size_t bsz, result = 0;
	size_t in_len;
	iconv_err_type ret = ICONV_ERR_SUCCESS;

	cd = iconv_open(out_charset, in_charset);

	if (cd == (iconv_t)(-1)) {
		if (errno == EINVAL) {
			return ICONV_ERR_WRONG_CHARSET;
		} else {
			return ICONV_ERR_CONVERTER;
		}
	}
	in_p = in_str;
	in_len = strlen(in_str);
	in_left= in_len;
	out_left = in_len + 32; /* Avoid realloc() most cases */ 
	out_size = 0;
	bsz = out_left;
	out_buf = (char *)calloc(bsz + 1, sizeof(char));
	out_p = out_buf;

	while (in_left > 0) {
		result = iconv(cd, (char **)&in_p, &in_left, (char **)&out_p, &out_left);
		out_size = bsz - out_left;
		if (result == (size_t)(-1)) {
			/* out_buf  ޸  ϰ ȯ  ̷ ʾҴٸ  ޸   */
			if (errno == E2BIG && in_left > 0) {
				/* converted string is longer than out buffer */
				bsz += in_len;
				tmp_buf = (char *)realloc(out_buf, bsz+1);
				out_p = out_buf = tmp_buf;
				out_p += out_size;
				out_left = bsz - out_size;
				continue;	
			}
		}
		break;
	}

	if (result != (size_t)(-1)) {
		/* flush the shift-out sequences */ 
		for (;;) {
		   	result = iconv(cd, NULL, NULL, (char **) &out_p, &out_left);
			out_size = bsz - out_left;

			if (result != (size_t)(-1)) {
				break;
			}

			if (errno == E2BIG) {
				bsz += 16;
				tmp_buf = (char *)realloc(out_buf, bsz);
				
				out_p = out_buf = tmp_buf;
				out_p += out_size;
				out_left = bsz - out_size;
			} else {
				break;
			}
		}
	}

	iconv_close(cd);

	if (result == (size_t)(-1)) {
		switch (errno) {
			/* Ϸ   Ʈ ڿ Է */
			case EINVAL:
				ret = ICONV_ERR_ILLEGAL_CHAR;
				break;

			/* Ȯ ߹Ʈ ڿ Է */
			case EILSEQ:
				ret = ICONV_ERR_ILLEGAL_SEQ;
				break;

			/* *outbuf   */
			case E2BIG:
				/* should not happen */
				ret = ICONV_ERR_TOO_BIG;
				break;

			default:
				/* other error */
				ret = ICONV_ERR_UNKNOWN;
				break;
		}
		free(out_buf);
	}
	*out_p = '\0';
	*out_str = out_buf;
	return ret;
}
#endif

void err_exit (char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);

	exit(FAILURE);
}

void http_request_free (http_request *hr)
{
        if (hr->host) free(hr->host);
        if (hr->uri) free(hr->uri);
        if (hr->rhdr) free(hr->rhdr);
}


void http_request_init (http_request *hr)
{
	hr->host = NULL;
	hr->uri = NULL;
	hr->rhdr = NULL;
}

char *http_request_get (char *url)
{
        int sock = 0x0;
        char buf[BUFSIZE];
        char *recved = NULL;
	memset(buf, 0x0, sizeof(buf));

	http_request *hr = (http_request *)malloc(sizeof(http_request));
	http_request_init(hr);

	if (parse_url((const char *)url, hr) != 0) {
		snprintf(buf, sizeof(buf) - 1, "parse_url(%s) Failed, %d", buf, __LINE__);
		perror(buf);
		goto try_return;
	}

	if ((sock = tcp_connect(hr->host, hr->port, SOTIMEO)) < 0) {
		snprintf(buf, sizeof(buf) - 1, "tcp_connect() Failed, %d Line, Host %s, Port %d", __LINE__, hr->host, hr->port);
		perror(buf);
		goto try_return;
	}

	socket_set_blocking (sock, NONSOCK);

	if (http_send_get(sock, hr) != 0) {
		snprintf(buf, sizeof(buf) - 1, "http_send_get() Failed, %d Line", __LINE__);
		perror(buf);
		goto try_return;
	}

	if (SUCCESS != http_recv_code(sock, "200")) {
		snprintf(buf, sizeof(buf) - 1, "http_recv_code() Failed, %d Line HTTP CODE != 200", __LINE__);
		perror(buf);
		goto try_return;
	}
	if (!(recved = http_recv_data(sock)))  {
		snprintf(buf, sizeof(buf) - 1, "http_recv_data() Failed, %d Line", __LINE__);
		perror(buf);
		goto try_return;
	}

try_return:
	http_request_free(hr);
	if (IS_SOCKET(sock)) socket_close(sock, LINGERON);

	return (recved) ? recved : (char *)NULL;
}

void dic_naver (char *str, config *cf)
{
	char buf[BUFSIZE];
	char *recved = NULL;
	char url[] = "http://endic.naver.com/small_search.nhn?kind=keyword&query=%s";
#ifdef HAVE_ICONV
	char *out_str = NULL;
#endif
	memset(buf, 0x0, sizeof(buf));
	snprintf(buf, sizeof(buf) - 1, url, str);

	if ((recved = http_request_get(buf))) {
#ifdef HAVE_ICONV
		str_toupper(cf->charset);
		if (strcmp(CDSET, cf->charset) != 0) {
			iconv_error(CDSET, cf->charset, iconv_convert (CDSET, cf->charset, recved, &out_str));
			free(recved);
			recved = out_str;
		}
#endif
		str_cut(recved, "<!-- ˻ -->", "<!--//˻-->");
		str_replace (recved, "<br>", "\n");
		strip_tag(recved);
		str_replace (recved, "\t", "");
		str_trim(recved);
		str_trims(recved);
		str_replace (recved, "\n\n\n", "");
		printf("%s\n", recved);
		free(recved);
	}
}

void dic_yahoo (char *str, config *cf)
{
	char buf[BUFSIZE];
	char *recved = NULL;
	char url[] = "http://kr.dictionary.search.yahoo.com/search/dictionarym?subtype=eng&p=%s";
#ifdef HAVE_ICONV
	char *out_str = NULL;
#endif
	memset(buf, 0x0, sizeof(buf));
	snprintf(buf, sizeof(buf) - 1, url, str);

	if ((recved = http_request_get(buf))) {
#ifdef HAVE_ICONV
		str_toupper(cf->charset);
		if (strcmp(CDSET, cf->charset) != 0) {
			iconv_error(CDSET, cf->charset, iconv_convert (CDSET, cf->charset, recved, &out_str));
			free(recved);
			recved = out_str;
		}
#endif
		str_cut(recved, "<!-- Begin-Module:db0 -->", "<!-- / -->");
		str_replace (recved, "<br>", "\n");
		strip_tag(recved);
		str_replace (recved, "\t", "");
		str_trim(recved);
		str_trims(recved);
		str_replace (recved, "\n\n\n", "");
		str_replace (recved, " \n", "");
		printf("%s\n", recved);
		free(recved);
	}

}

void dic_daum (char *str, config *cf)
{
	char buf[BUFSIZE];
	char *recved = NULL;
	char url[] = "http://engdic.daum.net/dicen/small_search_word.do?q=%s";
#ifdef HAVE_ICONV
	char *out_str = NULL;
#endif
	memset(buf, 0x0, sizeof(buf));
	snprintf(buf, sizeof(buf) - 1, url, str);

	if ((recved = http_request_get(buf))) {
#ifdef HAVE_ICONV
		str_toupper(cf->charset);
		if (strcmp(CDSET, cf->charset) != 0) {
			iconv_error(CDSET, cf->charset, iconv_convert (CDSET, cf->charset, recved, &out_str));
			free(recved);
			recved = out_str;
		}
#endif
		str_cut(recved, "<div id=\"contentSub_mini\">", "</div>");
		str_replace (recved, "<br>", "\n");
		strip_tag(recved);
		str_replace (recved, "\t", "");
		str_trim(recved);
		str_trims(recved);
		str_replace (recved, "\n\n\n", "");
		str_replace (recved, "\r\n\r\n", "");
		printf("%s\n", recved);
		free(recved);
	}
}

void dic_empas (char *str, config *cf)
{
	char buf[BUFSIZE];
	char *recved = NULL;
	char url[] = "http://alldic.empas.com/search/endic_sd.html?q=%s";
#ifdef HAVE_ICONV
	char *out_str = NULL;
#endif
	memset(buf, 0x0, sizeof(buf));
	snprintf(buf, sizeof(buf) - 1, url, str);

	if ((recved = http_request_get(buf))) {
#ifdef HAVE_ICONV
		str_toupper(cf->charset);
		if (strcmp(CDSET, cf->charset) != 0) {
			iconv_error(CDSET, cf->charset, iconv_convert (CDSET, cf->charset, recved, &out_str));
			free(recved);
			recved = out_str;
		}
#endif
		str_replace (recved, "<br>", "\n");
		strip_tag(recved);
		str_replace (recved, "\t", "");
		str_trim(recved);
		str_trims(recved);
		str_replace (recved, "\n\n\n", "");
		printf("%s\n", recved);
		free(recved);
	}

}

void trans_google (char *str, config *cf)
{
	int w = 0;
	char *ptr = NULL;
	char buf[BUFSIZE];
	char *recved = NULL, *lang_c;
	char url[] = "http://translate.google.com/translate_t?langpair=%s&ie=EUC-KR&text=%s";
#ifdef HAVE_ICONV
	char *out_str = NULL;
#endif

	ptr = strchr(str, ' ');
	w = (ptr) ? (ptr - str) : w;
	lang_c = (*str < 0) ? "ko|en" : "en|ko";	
	while (w--){
		if (*(str+w) < 0) {
			lang_c = "ko|en";
			break;
		}
	}

	memset(buf, 0x0, sizeof(buf));
	snprintf(buf, sizeof(buf) - 1, url, lang_c, str);

	if ((recved = http_request_get(buf))) {
#ifdef HAVE_ICONV
		str_toupper(cf->charset);
		if (strcmp(CDSET, cf->charset) != 0) {
			iconv_error(CDSET, cf->charset, iconv_convert (CDSET, cf->charset, recved, &out_str));
			free(recved);
			recved = out_str;
		}
#endif
		str_cut(recved, "<textarea name=utrans wrap=SOFT dir=ltr rows=5 id=suggestion>", "</textarea>");
		printf("%s\n", recved);
		free(recved);
	}
}

void trans_yahoo (char *str, config *cf)
{
	int w = 0;
	char *ptr = NULL;
	char buf[BUFSIZE];
	char *recved = NULL, *lang_c;
	char url[] = "http://kr.babelfish.yahoo.com/translate_txt?lp=%s&trtext=%s";
#ifdef HAVE_ICONV
	char *out_str = NULL;
#endif
	ptr = strchr(str, ' ');
	w = (ptr) ? (ptr - str) : w;
	lang_c = (*str < 0) ? "ko_en" : "en_ko";	
	while (w--){
		if (*(str+w) < 0) {
			lang_c = "ko_en";
			break;
		}
	}

	memset(buf, 0x0, sizeof(buf));
	snprintf(buf, sizeof(buf) - 1, url, lang_c, str);

	if ((recved = http_request_get(buf))) {
#ifdef HAVE_ICONV
		str_toupper(cf->charset);
		if (strcmp(CDSET, cf->charset) != 0) {
			iconv_error(CDSET, cf->charset, iconv_convert (CDSET, cf->charset, recved, &out_str));
			free(recved);
			recved = out_str;
		}
#endif
		str_cut(recved, "<div style=\"padding:0.6em;\">", "</div>");
		printf("%s\n", recved);
		free(recved);
	}
}

void config_init (config *cf)
{
	cf->dic = 0;
	cf->trans = 0;
#ifdef HAVE_ICONV
	memset(cf->charset, 0x0, sizeof(cf->charset));
	strncpy(cf->charset, CDSET, sizeof(cf->charset) - 1);
#endif
}

int config_load (char *path, config *cf)
{
	FILE *FP_IN = NULL;
	char *left, *right, buf[BUFSIZE>>1];

	config_init(cf);

	FP_IN = fopen(path, "r");
	if (FP_IN == NULL)
		return FAILURE;
	
	while (fgets(buf, sizeof(buf) - 1, FP_IN) != NULL) {

		str_replace (buf, " ", "");
		str_replace (buf, "\t", "");
		if (buf[0] ==  '#' || buf[0] ==  '\n')
			continue;

		if ((left = (char *)split(buf, '=', 0)) == NULL)
			continue;

		if ((right = (char *)strchr(buf, '=')) == NULL) {
			free(left);
			continue;
		}
		right++;

		str_trim(right);

		if (strcmp(left, "dic") == 0)
			 cf->dic = (dic_type)atoi(right);
		else if (strcmp(left, "trans") == 0)
			 cf->trans = (trans_type)atoi(right);
#ifdef HAVE_ICONV
		else if (strcmp(left, "charset") == 0) {
			memset(cf->charset, 0x0, sizeof(cf->charset));
			strncpy(cf->charset, right, sizeof(cf->charset) - 1);
		}
#endif

		free(left);
	}
	fclose(FP_IN);

	return SUCCESS;
}

int config_reload (char *var, config *cf)
{
	int i = 0, c = 0;
	char *left, *right;

	if ((left = (char *)split(var, '=', 0)) == NULL)
		return FAILURE;

	if ((right = (char *)strchr(var, '=')) == NULL) {
		return FAILURE;
	}
	right++;

	if (strcmp(left, "dic") == 0) {
		c = _COUNT(dic, dic_cmd);
		for ( i = 0; i < c; i++) {
			if (strcmp(right, dic[i].type_name) == 0) {
				printf("    : %s => %s", dic[cf->dic].type_name, dic[i].type_name);
				cf->dic = (dic_type)i;
				return SUCCESS;
			}
		}

	} else if (strcmp(left, "trans") == 0) {
		c = _COUNT(trans, trans_cmd);
		for ( i = 0; i < c; i++) {
			if (strcmp(right, trans[i].type_name) == 0) {
				printf("    : %s => %s", trans[cf->trans].type_name, trans[i].type_name);
				cf->trans = (trans_type)i;
				return SUCCESS;
			}
		}
	}
#ifdef HAVE_ICONV
	else if (strcmp(left, "charset") == 0) {
		printf("    : %s => %s", cf->charset, right);
		memset(cf->charset, 0x0, sizeof(cf->charset));
		strncpy(cf->charset, right, sizeof(cf->charset) - 1);
		return SUCCESS;
	}
#endif
	return FAILURE;
}

int config_create (char *path, config *cf)
{
	char buf[BUFSIZE>>1];
	FILE *FP_OUT = NULL;
	
	FP_OUT = fopen(path, "w");
	if (FP_OUT == NULL) {
		memset(buf, 0x0, sizeof(buf));
		snprintf(buf, sizeof(buf) - 1, "config_create %s", path);
		perror(buf);
		return FAILURE;
	}

	fprintf(FP_OUT,
			"# Dictionary select  [NAVER = 0, YAHOO = 1, DAUM = 2, EMPAS = 3]\n"
			"dic=%d\n\n"
			"# Translation select [GOOGLE_TRANS = 0, YAHOO_TRANS = 1]\n"
			"trans=%d\n\n"
#ifdef HAVE_ICONV
			"# Character set select [EUC-KR, UTF-8 ...] ( iconv --list )\n"
			"# The iconv -l or iconv --list command lists the names of the supported encodings, in a system dependent format\n"
			"charset=%s\n"
#endif
			, cf->dic
			, cf->trans
#ifdef HAVE_ICONV
			, cf->charset
#endif
	);
	fclose(FP_OUT);

	return SUCCESS;
}

char *ansi_color (char *str, unsigned int color, unsigned int bold)
{
	int len = 0;
	char *ptr = NULL;
	char ac[] = "\033[%d;%dm%s\033[0m";

	len = strlen(str) + strlen(ac);
	ptr = (char *)calloc(len, sizeof(char));
	bold = (bold|0)&1;

	switch(color) {

		case GRAY:
		case RED:
		case GREEN:
		case YELLOW:
		case BLUE:
		case MAGENTA:
		case CYAN:
		case WHITE:
		case NOT:
			snprintf(ptr, len - 1, ac, bold, color, str);
			break;
		default :
			snprintf(ptr, len - 1, ac, bold, NOT, str);
			break;
	}

	return ptr;
}


void usage (const char *prog, config *cf)
{
	int  i= 0, c = 0;
	char *check = NULL, *check_name = NULL;
	FILE* stream;
	stream = (!prog) ? stderr : stdout;

	fprintf(stream, "CDIC %s By YoungJoo-Kim <bando@bando.org>\n\n"
			"Usage : %s [ܾ] or []\n\n"
			"\t[  ]\n"
			, PROG_VERSION, prog
	);

	c = _COUNT(dic, dic_cmd);
	for ( i = 0; i < c; i++) {
		check = (cf->dic == i) ? ansi_color("*", GREEN, UNBOLD) : ansi_color(" ", NOT, UNBOLD);	
		check_name = ansi_color(dic[i].type_name, MAGENTA, UNBOLD);
		printf("\t\t%-30s[%s]\n", check_name, check );
		if (check) free(check);
		if (check_name) free(check_name);
	}
	
	fprintf(stream, "\n\t[  ]\n");

	c = _COUNT(trans, trans_cmd);
	for ( i = 0; i < c; i++) {
		check = (cf->trans == i) ? ansi_color("*", GREEN, UNBOLD) : ansi_color(" ", NOT, UNBOLD);	
		check_name = ansi_color(trans[i].type_name, MAGENTA, UNBOLD);
		printf("\t\t%-30s[%s]\n", check_name, check );
		if (check) free(check);
		if (check_name) free(check_name);
	}

#ifdef HAVE_ICONV
	fprintf(stream, "\n\t[  ]\n");
	check = ansi_color(cf->charset, GREEN, UNBOLD);	
	check_name = ansi_color("CHARSET", MAGENTA, UNBOLD);
	printf("\t\t%-30s[%s]\n", check_name, check );
	if (check) free(check);
	if (check_name) free(check_name);
#endif

	fprintf(stream, "\n\t[  ϱ]\n"
			"\t\tcdic dic=YAHOO\n\n"
			"\t[  ϱ]\n"
			"\t\tcdic trans=YAHOO_TRANS\n\n"
#ifdef HAVE_ICONV
			"\t[  ϱ]\n"
			"\t\tcdic charset=[EUC-KR|UTF-8...]\n\n"
#endif
	);
		
	fflush(stream);

	exit(1);
}

